package presentation;

/**
 * @author Alberto Iachini
 */

import java.awt.GridBagLayout;

import javax.swing.JFileChooser;
import javax.swing.JPanel;
import javax.swing.JButton;
import java.awt.GridBagConstraints;
import javax.swing.JLabel;
import java.awt.Insets;
import java.io.File;
import presentation.graph.GraphFactory;
import presentation.util.Exporter;
import control.Orchestrator;
import org.jgraph.JGraph;
import javax.swing.JScrollPane;
import javax.swing.filechooser.FileNameExtensionFilter;

import abstraction.Service;

/**
 * This class is the panel used to show the graph of the orchestrator
 */
public class OrchestratorGraphPanel extends JPanel {

	private static final long serialVersionUID = 971167226552503028L;
	
	private OrchestratorFrame owner = null;

	private JButton changeRepresentationButton = null;
	private JButton saveToImageButton = null;
	private JButton saveToDotButton = null;
	private JLabel orchestratorLabel = null;
	private JScrollPane orchestratorScrollPane = null;
	
	private JGraph orchestratorGraph = null;
	private Service orchestrator = null; 
	

	/**
	 * This is the default constructor
	 */
	public OrchestratorGraphPanel(OrchestratorFrame frame) {
		super();
		initialize();
		owner = frame;
	}

	/**
	 * This method initializes this
	 */
	private void initialize() {
		GridBagConstraints gridBagConstraints12 = new GridBagConstraints();
		gridBagConstraints12.gridx = 0;
		gridBagConstraints12.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints12.ipadx = 14;
		gridBagConstraints12.gridy = 4;
		GridBagConstraints gridBagConstraints5 = new GridBagConstraints();
		gridBagConstraints5.fill = GridBagConstraints.BOTH;
		gridBagConstraints5.gridy = 1;
		gridBagConstraints5.weightx = 10.0;
		gridBagConstraints5.weighty = 10.0;
		gridBagConstraints5.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints5.gridheight = 6;
		gridBagConstraints5.gridx = 1;
		GridBagConstraints gridBagConstraints2 = new GridBagConstraints();
		gridBagConstraints2.gridx = 0;
		gridBagConstraints2.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints2.anchor = GridBagConstraints.WEST;
		gridBagConstraints2.ipadx = 38;
		gridBagConstraints2.gridy = 3;
		GridBagConstraints gridBagConstraints1 = new GridBagConstraints();
		gridBagConstraints1.gridx = 0;
		gridBagConstraints1.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints1.anchor = GridBagConstraints.WEST;
		gridBagConstraints1.ipadx = 43;
		gridBagConstraints1.gridy = 0;
		orchestratorLabel = new JLabel();
		orchestratorLabel.setText("Orchestrator");
		GridBagConstraints gridBagConstraints = new GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints.anchor = GridBagConstraints.WEST;
		gridBagConstraints.ipadx = 49;
		gridBagConstraints.gridy = 2;
		this.setSize(300, 200);
		this.setLayout(new GridBagLayout());
		this.add(getSaveToImageButton(), gridBagConstraints);
		this.add(orchestratorLabel, gridBagConstraints1);
		this.add(getSaveToDotButton(), gridBagConstraints2);
		this.add(getOrchestratorScrollPane(), gridBagConstraints5);
		this.add(getChangeRepresentationButton(), gridBagConstraints12);
	}
	
	/**
	 * Initialize the orchestrator panel with the generated orchestator
	 * 
	 * @param orch the orchestrator
	 */
	public void init(Orchestrator orch)
	{
		orchestratorGraph = null;
		orchestratorScrollPane.setViewportView(null);
		orchestrator = orch.createOrchestratorGenerator();
		if(orchestrator != null)
			orchestratorGraph = (new GraphFactory(orchestrator)).getGraph();
		orchestratorScrollPane.setViewportView(orchestratorGraph);
	}

	/**
	 * This method initializes saveToImageButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSaveToImageButton() {
		if (saveToImageButton == null) {
			saveToImageButton = new JButton();
			saveToImageButton.setText("Save To Image");
			saveToImageButton.setToolTipText("Save the graph in a jpeg file");
			saveToImageButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					if(orchestratorGraph != null)
					{
						File file = saveImageFile();
						if(file != null)
							Exporter.exportToImage(file, orchestratorGraph);
					}
				}
			});
		}
		return saveToImageButton;
	}

	/**
	 * This method initializes saveToDotButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSaveToDotButton() {
		if (saveToDotButton == null) {
			saveToDotButton = new JButton();
			saveToDotButton.setText("Save To DOT File");
			saveToDotButton.setToolTipText("Save the graph in DOT format");
			saveToDotButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					if(orchestrator != null)
					{
						File file = saveDotFile();
						if(file != null)
							Exporter.exportToDot(file, orchestrator);
					}
				}
			});
		}
		return saveToDotButton;
	}


	/**
	 * This method initializes orchestratorScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getOrchestratorScrollPane() {
		if (orchestratorScrollPane == null) {
			orchestratorScrollPane = new JScrollPane();
			orchestratorScrollPane.setViewportView(null);
		}
		return orchestratorScrollPane;
	}

	/**
	 * Open a JFileChooser to save the selected graph into jpeg file
	 * 
	 * @return the selected file
	 */
	private File saveImageFile()
	{
		JFileChooser fc = new JFileChooser();
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		FileNameExtensionFilter filter = new FileNameExtensionFilter("JPEG File", "jpeg");
		fc.setFileFilter(filter);
		fc.showSaveDialog(this);
		owner.setLastFolder(fc.getCurrentDirectory());
		return fc.getSelectedFile();
	}
	
	/**
	 * Open a JFileChooser to save the selected graph into dot file
	 * 
	 * @return the selected file
	 */
	private File saveDotFile()
	{
		JFileChooser fc = new JFileChooser();
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		FileNameExtensionFilter filter = new FileNameExtensionFilter("DOT File", "dot");
		fc.setFileFilter(filter);
		fc.showSaveDialog(this);
		owner.setLastFolder(fc.getCurrentDirectory());
		return fc.getSelectedFile();
	}

	/**
	 * This method initializes changeRepresentationButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getChangeRepresentationButton() {
		if (changeRepresentationButton == null) {
			changeRepresentationButton = new JButton();
			changeRepresentationButton.setText("Change Visualization");
			changeRepresentationButton.setToolTipText("Change the layout of the graph");
			changeRepresentationButton
					.addActionListener(new java.awt.event.ActionListener() {
						public void actionPerformed(java.awt.event.ActionEvent e) {
							if(orchestratorGraph != null)
							{
								ChoiceLayoutDialog dialog = new ChoiceLayoutDialog(owner);
								String layout = dialog.getGraphLayout();
								if(layout != null)
								{
									GraphFactory.changeRepresentation(orchestratorGraph, layout);
								}
							}
						}
					});
		}
		return changeRepresentationButton;
	}

}
